<?php
// create_location.php - Minor UTF-8 improvements

header('Content-Type: application/json; charset=utf-8');

$config = include_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../stripe/vendor/autoload.php';

$key = $config['stripe_key'];

if (!$key) {
    http_response_code(500);
    echo json_encode(['error' => 'Stripe configuration error: Unable to retrieve secret key.']);
    exit;
}

$accessToken = $key;

// Set default location data
$location_data = [
    'display_name' => 'Main Box Office',
    'address[line1]' => '514 G St SW',
    'address[city]' => 'Washington',
    'address[state]' => 'DC',
    'address[postal_code]' => '20024',
    'address[country]' => 'US'
];

// Optionally override with POST data
foreach ($location_data as $key => $default) {
    if (isset($_POST[$key])) {
        // Clean UTF-8 on input
        $location_data[$key] = mb_convert_encoding($_POST[$key], 'UTF-8', 'UTF-8');
    }
}

// Create location via Stripe API
$ch = curl_init();
curl_setopt_array($ch, [
    CURLOPT_URL => 'https://api.stripe.com/v1/terminal/locations',
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => http_build_query($location_data),
    CURLOPT_HTTPHEADER => [
        "Authorization: Bearer $accessToken"
    ],
    CURLOPT_TIMEOUT => 30,
    CURLOPT_CONNECTTIMEOUT => 10
]);

$response = curl_exec($ch);
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$curl_error = curl_error($ch);
curl_close($ch);

// Handle cURL errors
if ($curl_error) {
    http_response_code(500);
    echo json_encode(['error' => "Network error: $curl_error"], JSON_UNESCAPED_UNICODE);
    exit;
}

// Parse response
$response_data = json_decode($response, true);

// Return result with proper error handling
if ($http_code >= 200 && $http_code < 300) {
    // Success
    if ($response_data && isset($response_data['id'])) {
        echo json_encode([
            'id' => $response_data['id'],
            'display_name' => $response_data['display_name'],
            'success' => true
        ], JSON_UNESCAPED_UNICODE);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Invalid response from Stripe API'], JSON_UNESCAPED_UNICODE);
    }
} else {
    // Error
    http_response_code($http_code);
    
    if ($response_data && isset($response_data['error'])) {
        // Return Stripe's error message
        echo json_encode([
            'error' => $response_data['error']['message'] ?? 'Stripe API error',
            'type' => $response_data['error']['type'] ?? 'api_error'
        ], JSON_UNESCAPED_UNICODE);
    } else {
        // Fallback error
        echo json_encode([
            'error' => "HTTP error: $http_code",
            'response' => $response
        ], JSON_UNESCAPED_UNICODE);
    }
}