<?php
// box/api/customers.php

// Clear any potential output buffers before loading config
while (ob_get_level()) {
    ob_end_clean();
}

// Load config
$config = include __DIR__ . '/../config/config.php';

// Set JSON header with UTF-8 charset
header('Content-Type: application/json; charset=utf-8');

// Database connection
$conn = new mysqli(
    $config['db_host'],
    $config['db_user'],
    $config['db_pass'],
    $config['db_name']
);

// Check connection
if ($conn->connect_error) {
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed: ' . $conn->connect_error]);
    exit;
}

// ✅ CRITICAL: Force UTF-8 connection
$conn->set_charset('utf8mb4');

// Configuration
$max_customers = 1000;

// Check which database we're in
$is_ncetickets = (strpos(strtolower($config['db_name']), 'ncetickets') !== false);

if ($is_ncetickets) {
    // NCETickets database - simple users table
    $sql = "SELECT id, name, email FROM users WHERE email IS NOT NULL AND email != '' AND name IS NOT NULL AND name != '' ORDER BY id DESC LIMIT ?";
    
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        http_response_code(500);
        echo json_encode(['error' => 'Query preparation failed', 'details' => $conn->error]);
        exit;
    }
    $stmt->bind_param("i", $max_customers);
} else {
    // osConcert database - customers table
    $sql = "SELECT customers_id as id, CONCAT(IFNULL(customers_firstname, ''), ' ', IFNULL(customers_lastname, '')) as name, customers_email_address as email FROM customers WHERE customers_email_address IS NOT NULL AND customers_email_address != '' ORDER BY customers_id DESC LIMIT ?";
    
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        http_response_code(500);
        echo json_encode(['error' => 'Query preparation failed', 'details' => $conn->error]);
        exit;
    }
    $stmt->bind_param("i", $max_customers);
}

$stmt->execute();
$result = $stmt->get_result();

$customers = [];

while ($row = $result->fetch_assoc()) {
    // Clean UTF-8 encoding issues
    $name = $row['name'] ?? '';
    $email = $row['email'] ?? '';
    
    // Remove any non-UTF8 characters
    $name = mb_convert_encoding($name, 'UTF-8', 'UTF-8');
    $email = mb_convert_encoding($email, 'UTF-8', 'UTF-8');
    
    // Also try iconv as a fallback
    if (function_exists('iconv')) {
        $name = iconv('UTF-8', 'UTF-8//IGNORE', $name);
        $email = iconv('UTF-8', 'UTF-8//IGNORE', $email);
    }
    
    // Trim and clean
    $name = trim($name);
    if (empty($name) || $name === ' ') {
        $name = 'Customer ' . $row['id'];
    }
    
    $customers[] = [
        'id' => (int)$row['id'],
        'name' => $name,
        'email' => strtolower(trim($email))
    ];
}

$stats = [
    'customer_count' => count($customers),
    'max_customers' => $max_customers,
    'database' => $config['db_name']
];

$response = [
    'customers' => $customers,
    'stats' => $stats
];

$stmt->close();
$conn->close();

// Encode with UTF-8 support
$json_output = json_encode($response, JSON_UNESCAPED_UNICODE);
if ($json_output === false) {
    http_response_code(500);
    echo json_encode(['error' => 'JSON encoding failed: ' . json_last_error_msg()]);
    exit;
}

// Output the JSON
echo $json_output;

// Make sure nothing else outputs
exit();