<?php
// box/api/misc/view.php - Backend for misc orders viewer
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type');

// Get filter parameters
$dateFrom = $_GET['dateFrom'] ?? '';
$dateTo = $_GET['dateTo'] ?? '';
$customer = $_GET['customer'] ?? '';
$orderType = $_GET['orderType'] ?? '';

// Read the log file
$logFile = __DIR__ . '/misc_orders.log';
$orders = [];

if (file_exists($logFile)) {
    $lines = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    
    foreach ($lines as $line) {
        $order = json_decode($line, true);
        if ($order) {
            // Convert old format entries to new format for compatibility
            if (!isset($order['financial_breakdown']) && isset($order['amount'])) {
                // Old format - convert to new format
                $order = [
                    'order_id' => $order['order_id'] ?? null,
                    'timestamp' => $order['date'] ?? date('Y-m-d H:i:s'),
                    'order_uuid' => uniqid(),
                    'transaction_id' => $order['payment_method'] ?? 'unknown',
                    'customer_name' => $order['customer'] ?? 'Unknown',
                    'customer_email' => '',
                    'order_type' => $order['type'] ?? 'unknown',
                    'items' => [
                        [
                            'name' => $order['items'] ?? 'Unknown Items',
                            'quantity' => 1,
                            'price' => $order['amount'] ?? 0,
                            'total' => $order['amount'] ?? 0
                        ]
                    ],
                    'financial_breakdown' => [
                        'subtotal' => $order['amount'] ?? 0,
                        'tax_amount' => 0,
                        'tip_amount' => 0,
                        'donation_amount' => 0,
                        'discount_amount' => 0,
                        'applied_coupon' => null,
                        'total' => $order['amount'] ?? 0
                    ]
                ];
            }
            
            $orders[] = $order;
        }
    }
}

// Apply filters
$filteredOrders = array_filter($orders, function($order) use ($dateFrom, $dateTo, $customer, $orderType) {
    // Date filtering
    if ($dateFrom) {
        $orderDate = date('Y-m-d', strtotime($order['timestamp']));
        if ($orderDate < $dateFrom) {
            return false;
        }
    }
    
    if ($dateTo) {
        $orderDate = date('Y-m-d', strtotime($order['timestamp']));
        if ($orderDate > $dateTo) {
            return false;
        }
    }
    
    // Customer filtering
    if ($customer) {
        $customerLower = strtolower($customer);
        $nameMatch = stripos($order['customer_name'], $customer) !== false;
        $emailMatch = stripos($order['customer_email'] ?? '', $customer) !== false;
        if (!$nameMatch && !$emailMatch) {
            return false;
        }
    }
    
    // Order type filtering
    if ($orderType && $order['order_type'] !== $orderType) {
        return false;
    }
    
    return true;
});

// Sort by timestamp (newest first)
usort($filteredOrders, function($a, $b) {
    return strtotime($b['timestamp']) - strtotime($a['timestamp']);
});

// Return the filtered results
echo json_encode(array_values($filteredOrders));
?>