<?php
// box/api/products.php - Adapted for osConcert database


// Clean any output before headers
ob_start();

// ---------------------------------------------------------------------
// Define $topDir via getTopDir.php (single source of truth)
// ---------------------------------------------------------------------
require_once __DIR__ . '/../config/getTopDir.php';

$topDir = getTopDir();

// Load osConcert database config 
$config = include __DIR__ . '/../config/config.php';

// Clear output buffer before sending headers
ob_end_clean();

// Set JSON header with UTF-8 charset
header('Content-Type: application/json; charset=utf-8');

// Database connection using osConcert config - SINGLE CONNECTION
$conn = new mysqli(
    $config['db_host'],
    $config['db_user'],
    $config['db_pass'],
    $config['db_name']
);

// Check connection
if ($conn->connect_error) {
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed: ' . $conn->connect_error]);
    exit;
}

// ✅ CRITICAL: Force UTF-8 connection
$conn->set_charset('utf8mb4');

// Fetch active events with ticket products from osConcert structure
$sql = "
    SELECT 
        p.products_id,
        p.products_price,
        p.products_quantity,
        p.products_date_available,
        pd.products_name,
        ptc.categories_id,
        cd.categories_name AS event_name,
        cd.concert_venue,
        c.categories_image
    FROM products p
    INNER JOIN products_description pd ON p.products_id = pd.products_id
    INNER JOIN products_to_categories ptc ON p.products_id = ptc.products_id
    INNER JOIN categories c ON ptc.categories_id = c.categories_id
    INNER JOIN categories_description cd ON c.categories_id = cd.categories_id
    WHERE p.products_status < '2'
    AND c.categories_status = 1
    AND p.products_quantity > 0
    AND pd.language_id = 1
    AND cd.language_id = 1
    ORDER BY p.products_id DESC
";

$result = $conn->query($sql);

if (!$result) {
    http_response_code(500);
    echo json_encode(['error' => 'Query failed', 'mysql_error' => $conn->error]);
    exit;
}

$products = [];

while ($row = $result->fetch_assoc()) {
    // Clean UTF-8 data to prevent JSON encoding errors
    $venue = mb_convert_encoding($row['concert_venue'] ?? '', 'UTF-8', 'UTF-8');
    $event_name = mb_convert_encoding($row['event_name'] ?? '', 'UTF-8', 'UTF-8');
    $ticket_name = mb_convert_encoding($row['products_name'] ?? '', 'UTF-8', 'UTF-8');
    
    // Get event date - use products_date_available as fallback
    $event_date = '';
    if (!empty($row['products_date_available'])) {
        $event_date = $row['products_date_available'];
    }
    
    // Build image URL using categories_image
    $image_url = null;
    if (!empty($row['categories_image'])) {
        $image_url = $topDir . 'images/big/' . $row['categories_image'];
    }
    
    $products[] = [
        'id'         => (int)$row['products_id'],
        'event_id'   => (int)$row['categories_id'],
        'ticket_name'=> $ticket_name,
        'price'      => (float)$row['products_price'],
        'event_name' => $event_name,
        'venue'      => $venue,
        'date'       => $event_date,
        'image_url'  => $image_url
    ];
}

// Close connection once at the end
$conn->close();

// Return the products with proper UTF-8 encoding
$json_output = json_encode($products, JSON_UNESCAPED_UNICODE);
if ($json_output === false) {
    http_response_code(500);
    echo json_encode(['error' => 'JSON encoding failed: ' . json_last_error_msg()]);
    exit;
}

echo $json_output;