<?php
// box/api/record_full_order.php - Complete osConcert-compatible order writer
// Handles both Android and iOS POS apps

ob_start();
$config = include __DIR__ . '/../config/config.php';
ob_end_clean();

header('Content-Type: application/json; charset=utf-8');

$conn = new mysqli($config['db_host'], $config['db_user'], $config['db_pass'], $config['db_name']);

if ($conn->connect_error) {
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed']);
    exit;
}

$conn->set_charset('utf8mb4');

// Get JSON input
$raw_input = file_get_contents('php://input');
$data = json_decode($raw_input, true);

if (!$data || !isset($data['items'], $data['total_amount'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid request data']);
    exit;
}

// Extract order data with defaults
$items = $data['items'];
$total_amount = (float)$data['total_amount'];
$subtotal = (float)($data['subtotal'] ?? 0);
$tax_amount = (float)($data['tax_amount'] ?? 0);
$tip_amount = (float)($data['tip_amount'] ?? 0);
$donation_amount = (float)($data['donation_amount'] ?? 0);
$discount_amount = (float)($data['discount_amount'] ?? 0);
$applied_coupon = trim($data['applied_coupon'] ?? '');
$transaction_id = trim($data['transaction_id'] ?? 'pos_' . time());
$order_id_uuid = $data['order_id'] ?? uniqid('order_');

// Customer info with Box Office defaults
$customer_name_raw = trim($data['customer_name'] ?? 'Box Office');
$customer_email_raw = trim($data['customer_email'] ?? '');
$customer_name = mb_convert_encoding($customer_name_raw, 'UTF-8', 'UTF-8');
$customer_email = mb_convert_encoding($customer_email_raw, 'UTF-8', 'UTF-8');

// Payment method
$payment_method_raw = $data['payment_method'] ?? 'card';
$payment_method = ($payment_method_raw === 'cash') ? 'Box Office Staff' : 'Stripe Secure Payments';

// Mark all POS orders with consistent billing name
$billing_name = "Box Office POS";

// Separate ticket items from misc items
$ticket_items = array_filter($items, function($item) {
    return isset($item['isRemote']) && $item['isRemote'] === true;
});

$misc_items = array_filter($items, function($item) {
    return !isset($item['isRemote']) || $item['isRemote'] === false;
});

$has_tickets = !empty($ticket_items);
$has_misc = !empty($misc_items);

// Calculate ticket subtotal
$ticket_subtotal = 0;
foreach ($ticket_items as $item) {
    $ticket_subtotal += (float)$item['price'] * (int)$item['quantity'];
}

// Calculate misc subtotal
$misc_subtotal = 0;
foreach ($misc_items as $item) {
    $misc_subtotal += (float)$item['price'] * (int)$item['quantity'];
}

// If no tickets and no misc, error
if (!$has_tickets && !$has_misc) {
    http_response_code(400);
    echo json_encode(['error' => 'No items found']);
    exit;
}

$response = ['success' => true];
$order_id = null;

// ONLY create database order if there are tickets
if ($has_tickets) {
    // Calculate ticket portion of financial breakdown
    $ticket_proportion = $subtotal > 0 ? $ticket_subtotal / $subtotal : 1;
    $ticket_tax = $tax_amount * $ticket_proportion;
    $ticket_discount = $discount_amount * $ticket_proportion;
    
    // For mixed orders: tickets get their portion, no tip/donation
    // For ticket-only: tickets get everything including tip+donation
    if ($has_misc) {
        $ticket_total = $ticket_subtotal + $ticket_tax - $ticket_discount;
        $ticket_tip = 0;
        $ticket_donation = 0;
    } else {
        $ticket_total = $ticket_subtotal + $ticket_tax + $tip_amount + $donation_amount - $ticket_discount;
        $ticket_tip = $tip_amount;
        $ticket_donation = $donation_amount;
    }

// Find or create customer
$customer_id = null;
$customer_address = null; // Will hold address data if found

if ($customer_email) {
    // Customer provided email - look up existing customer
    $stmt = $conn->prepare("
        SELECT c.customers_id, c.customers_firstname, c.customers_lastname, 
               c.customers_telephone, c.customers_default_address_id,
               a.entry_street_address, a.entry_city, a.entry_state, 
               a.entry_postcode, a.entry_country_id, a.entry_zone_id,
               z.zone_code
        FROM customers c 
        LEFT JOIN address_book a ON c.customers_id = a.customers_id 
            AND c.customers_default_address_id = a.address_book_id
        LEFT JOIN zones z ON a.entry_zone_id = z.zone_id
        WHERE c.customers_email_address = ? 
        LIMIT 1
    ");
    $stmt->bind_param("s", $customer_email);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $customer_id = (int)$row['customers_id'];
        // Store address info if available
        // Use zone_code if available, otherwise fall back to entry_state
        $state = $row['zone_code'] ?? $row['entry_state'] ?? '';
        
        $customer_address = [
            'street' => $row['entry_street_address'] ?? '',
            'city' => $row['entry_city'] ?? '',
            'state' => $state,
            'postcode' => $row['entry_postcode'] ?? '',
            'country' => 'United States', // Default
            'telephone' => $row['customers_telephone'] ?? ''
        ];
    }
    $stmt->close();
    
    // If email provided but customer doesn't exist, create new guest customer
    if (!$customer_id) {
        $parts = explode(' ', $customer_name, 2);
        $firstname = $parts[0] ?? 'Guest';
        $lastname = $parts[1] ?? 'Customer';
        
        $customer_sql = "INSERT INTO customers (
            customers_firstname, customers_lastname, customers_email_address,
            customers_telephone, customers_dob, customers_gender, customers_newsletter,
            customers_default_address_id, guest_account
        ) VALUES (?, ?, ?, '', '0000-00-00', '', '0', 0, 1)";
        
        $stmt = $conn->prepare($customer_sql);
        $stmt->bind_param("sss", $firstname, $lastname, $customer_email);
        $stmt->execute();
        $customer_id = $stmt->insert_id;
        $stmt->close();
    }
} else {
    // No email provided - use standard Box Office customer account (id=2)
    // This matches osConcert web box office behavior
    $customer_id = 2;
}

// Prepare address fields for order
$order_street = '';
$order_city = '';
$order_state = '';
$order_postcode = '';
$order_country = 'United States';
$order_telephone = '';

// Use customer address if found
if ($customer_address) {
    $order_street = $customer_address['street'];
    $order_city = $customer_address['city'];
    $order_state = $customer_address['state'];
    $order_postcode = $customer_address['postcode'];
    $order_country = $customer_address['country'];
    $order_telephone = $customer_address['telephone'];
}

// Create main order record - matching osConcert structure exactly
$order_sql = "INSERT INTO orders (
    customers_id, customers_name, customers_company, customers_street_address, 
    customers_suburb, customers_city, customers_postcode, customers_state, 
    customers_country, customers_telephone, customers_email_address,
    customers_address_format_id, customers_dummy_account,
    delivery_name, delivery_company, delivery_email, delivery_street_address, 
    delivery_suburb, delivery_city, delivery_postcode, delivery_state, 
    delivery_country, delivery_phone, delivery_address_format_id,
    billing_name, billing_company, billing_email, billing_street_address, 
    billing_suburb, billing_city, billing_postcode, billing_state, 
    billing_country, billing_phone, billing_address_format_id,
    payment_method, currency, currency_value, 
    date_purchased, last_modified, orders_status, orders_date_finished,
    ticket_printed, reference_id, payment_return2, 
    ip_address, customers_language
) VALUES (
    ?, ?, '', ?, '', ?, ?, ?, 
    ?, ?, ?,
    1, 0,
    '', '', '', '', '', '', '', '', 
    'United States', '', 1,
    ?, '', '', '', '', '', '', '', 
    'United States', '', 1,
    ?, 'USD', 1.000000,
    NOW(), NOW(), 3, NOW(),
    'Y', ?, ?,
    '', '1'
)";

$stmt = $conn->prepare($order_sql);
$stmt->bind_param("issssssssssss",
    $customer_id, $customer_name, 
    $order_street, $order_city, $order_postcode, $order_state, $order_country, $order_telephone, $customer_email,
    $billing_name, $payment_method, $transaction_id, $transaction_id
);
$stmt->execute();
$order_id = $stmt->insert_id;
$stmt->close();

// Helper function to format concert date
function formatConcertDate($date_str) {
    // Convert "08-11-2025" to "Saturday, November 8th"
    if (preg_match('/(\d{2})-(\d{2})-(\d{4})/', $date_str, $matches)) {
        $day = (int)$matches[1];
        $month = (int)$matches[2];
        $year = (int)$matches[3];
        $timestamp = mktime(0, 0, 0, $month, $day, $year);
        $day_name = date('l', $timestamp);
        $month_name = date('F', $timestamp);
        $day_suffix = date('jS', $timestamp);
        return "$day_name, $month_name $day_suffix";
    }
    return $date_str;
}

// Process ticket items
foreach ($ticket_items as $item) {
    // Get product ID - try productID first (both apps have it), then id as fallback
    $product_id = null;
    if (isset($item['productID'])) {
        $product_id = (int)$item['productID'];
    } elseif (isset($item['id']) && is_numeric($item['id'])) {
        $product_id = (int)$item['id'];
    }
    
    if (!$product_id) {
        error_log("Missing product ID in item: " . json_encode($item));
        continue;
    }
    
    $event_id = (int)($item['eventID'] ?? 0);
    $quantity = (int)$item['quantity'];
    $product_price = (float)$item['price'];
    $product_name_raw = $item['name'];
    $product_name = mb_convert_encoding($product_name_raw, 'UTF-8', 'UTF-8');
    
    // Lookup product details
    $stmt = $conn->prepare("SELECT products_model, products_date_available FROM products WHERE products_id = ?");
    $stmt->bind_param("i", $product_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $product_data = $result->fetch_assoc();
    $stmt->close();
    
    $products_model = $product_data['products_model'] ?? '';
    $products_date_available = $product_data['products_date_available'] ?? '2030-01-01 00:00:00';
    
    // Lookup event details
    $categories_name = '';
    $concert_venue = '';
    $concert_date = '';
    $concert_time = '';
    
    if ($event_id > 0) {
        $stmt = $conn->prepare("
            SELECT categories_name, concert_venue, concert_date, concert_time 
            FROM categories_description 
            WHERE categories_id = ? AND language_id = 1
        ");
        $stmt->bind_param("i", $event_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($event_data = $result->fetch_assoc()) {
            $categories_name = mb_convert_encoding($event_data['categories_name'], 'UTF-8', 'UTF-8');
            $concert_venue = mb_convert_encoding($event_data['concert_venue'], 'UTF-8', 'UTF-8');
            $concert_date = formatConcertDate($event_data['concert_date']);
            $concert_time = strtolower($event_data['concert_time']); // 7:30 PM -> 7:30 pm
        }
        $stmt->close();
    }
    
    // Calculate tax for this line item
    $line_total = $product_price * $quantity;
    $item_tax = $ticket_subtotal > 0 ? ($tax_amount * ($line_total / $ticket_subtotal)) : 0;
    
    // Insert complete orders_products record matching osConcert format
    $order_products_sql = "INSERT INTO orders_products (
        orders_id, products_id, products_model, products_name, 
        products_price, final_price, products_tax, products_quantity,
        products_type, events_type, support_packs_type,
        idcards_printed, is_printable,
        events_id, categories_name, concert_venue, concert_date, concert_time,
        products_date_available
    ) VALUES (
        ?, ?, ?, ?,
        ?, ?, ?, ?,
        'P', 'G', 'G',
        'N', 1,
        ?, ?, ?, ?, ?,
        ?
    )";
    
    $stmt = $conn->prepare($order_products_sql);
    $stmt->bind_param("iissdddiisssss",
        $order_id, $product_id, $products_model, $product_name,
        $product_price, $product_price, $item_tax, $quantity,
        $event_id, $categories_name, $concert_venue, $concert_date, $concert_time,
        $products_date_available
    );
    $stmt->execute();
    $stmt->close();
    
    // Update inventory
    $stmt = $conn->prepare("UPDATE products SET products_quantity = products_quantity - ? WHERE products_id = ?");
    $stmt->bind_param("ii", $quantity, $product_id);
    $stmt->execute();
    $stmt->close();
}

// Insert order totals matching osConcert format
$order_totals = [];

// Subtotal
$order_totals[] = [
    'title' => 'SubTotal:',
    'text' => '$' . number_format($ticket_subtotal, 2) . ' ',
    'value' => $ticket_subtotal,
    'class' => 'ot_subtotal',
    'sort_order' => 50
];

// Tax (if any)
if ($tax_amount > 0) {
    $order_totals[] = [
        'title' => 'Tax:',
        'text' => '$' . number_format($tax_amount, 2) . ' ',
        'value' => $tax_amount,
        'class' => 'ot_tax',
        'sort_order' => 60
    ];
}

// Discount (if coupon applied)
if ($discount_amount > 0 && $applied_coupon) {
    $order_totals[] = [
        'title' => 'Discount (' . $applied_coupon . '):',
        'text' => '-$' . number_format($discount_amount, 2) . ' ',
        'value' => -$discount_amount,
        'class' => 'ot_coupon',
        'sort_order' => 70
    ];
}

// Tip (if any)
if ($tip_amount > 0) {
    $order_totals[] = [
        'title' => 'Tip:',
        'text' => '$' . number_format($tip_amount, 2) . ' ',
        'value' => $tip_amount,
        'class' => 'ot_tip',
        'sort_order' => 80
    ];
}

// Donation (if any)
if ($donation_amount > 0) {
    $order_totals[] = [
        'title' => 'Donation:',
        'text' => '$' . number_format($donation_amount, 2) . ' ',
        'value' => $donation_amount,
        'class' => 'ot_donation',
        'sort_order' => 90
    ];
}

// Total
$order_totals[] = [
    'title' => 'Total:',
    'text' => '<b>$' . number_format($ticket_total, 2) . ' </b>',
    'value' => $ticket_total,
    'class' => 'ot_total',
    'sort_order' => 100
];

// Insert all order totals
foreach ($order_totals as $total) {
    $stmt = $conn->prepare("
        INSERT INTO orders_total (orders_id, title, text, value, class, sort_order)
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    $stmt->bind_param("issdsi",
        $order_id, $total['title'], $total['text'],
        $total['value'], $total['class'], $total['sort_order']
    );
    $stmt->execute();
    $stmt->close();
}

// Insert order status history
$stmt = $conn->prepare("
    INSERT INTO orders_status_history (orders_id, orders_status_id, date_added, comments)
    VALUES (?, 3, NOW(), 'Order processed via POS app')
");
$stmt->bind_param("i", $order_id);
$stmt->execute();
$stmt->close();

    $response['order_id'] = $order_id;
    $response['ticket_total'] = $ticket_total;
} // End of has_tickets block

// Process misc items if they exist
if ($has_misc) {
    // Calculate misc portion of financial breakdown
    if ($has_tickets) {
        // Mixed order: misc gets proportional tax/discount, no tip/donation
        $misc_proportion = $subtotal > 0 ? $misc_subtotal / $subtotal : 0;
        $misc_tax = $tax_amount * $misc_proportion;
        $misc_discount = $discount_amount * $misc_proportion;
        $misc_total = $misc_subtotal + $misc_tax - $misc_discount;
    } else {
        // Misc-only order: gets everything
        $misc_total = $misc_subtotal + $tax_amount + $tip_amount + $donation_amount - $discount_amount;
        $misc_tax = $tax_amount;
        $misc_discount = $discount_amount;
    }
    
    // Process misc items - create detailed log entry
    $item_details = [];
    foreach ($misc_items as $item) {
        $name_raw = trim($item['name'] ?? 'Unknown Item');
        $quantity = (int)($item['quantity'] ?? 1);
        $price = (float)($item['price'] ?? 0);
        $line_total = $price * $quantity;
        
        $name = mb_convert_encoding($name_raw, 'UTF-8', 'UTF-8');
        
        $item_details[] = [
            'name' => $name,
            'quantity' => $quantity,
            'price' => $price,
            'total' => $line_total
        ];
    }
    
    // Create comprehensive misc order log entry
    $misc_entry = [
        'order_id' => $order_id, // Will be null for misc-only orders
        'timestamp' => date('Y-m-d H:i:s'),
        'order_uuid' => $order_id_uuid,
        'transaction_id' => $transaction_id,
        'customer_name' => $customer_name,
        'customer_email' => $customer_email,
        'order_type' => $has_tickets ? 'mixed_order' : 'misc_only',
        'items' => $item_details,
        'financial_breakdown' => [
            'subtotal' => $misc_subtotal,
            'tax_amount' => $misc_tax,
            'tip_amount' => $has_tickets ? 0 : $tip_amount,
            'donation_amount' => $has_tickets ? 0 : $donation_amount,
            'discount_amount' => $misc_discount,
            'applied_coupon' => $applied_coupon,
            'total' => $misc_total
        ]
    ];
    
    // Write to misc orders log with UTF-8 support
    $misc_log_dir = __DIR__ . '/misc';
    if (!is_dir($misc_log_dir)) {
        mkdir($misc_log_dir, 0755, true);
    }
    file_put_contents($misc_log_dir . '/misc_orders.log', json_encode($misc_entry, JSON_UNESCAPED_UNICODE) . "\n", FILE_APPEND);
    
    $response['misc_total'] = $misc_total;
    $response['misc_items_count'] = count($item_details);
}

$conn->close();

echo json_encode($response, JSON_UNESCAPED_UNICODE);