<?php
// box/api/record_misc_order.php - Handle misc-only orders
header('Content-Type: application/json');

// Capture iPad JSON for debugging
$raw_input = file_get_contents('php://input');
file_put_contents('ipad_request_misc.json', $raw_input);

// Initialize debug log
$debug_log = "=== MISC ORDER DEBUG LOG - " . date('Y-m-d H:i:s') . " ===\n";

// Get JSON input from tablet
$data = json_decode($raw_input, true);
$debug_log .= "INPUT DATA: " . json_encode($data, JSON_PRETTY_PRINT) . "\n\n";

// Validate required fields for misc orders
if (!$data || !isset($data['items'], $data['total_amount'], $data['transaction_id'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid misc order request data']);
    exit;
}

// Extract misc order data
$items = $data['items'];
$total = (float)$data['total_amount'];
$transaction_id = trim($data['transaction_id']);
$customer_name = trim($data['customer_name'] ?? '');
$customer_email = trim($data['customer_email'] ?? '');
$title = trim($data['title'] ?? 'Misc Sale');
$description = trim($data['description'] ?? '');

$debug_log .= "EXTRACTED MISC DATA:\n";
$debug_log .= "Total: $total\n";
$debug_log .= "Transaction ID: $transaction_id\n";
$debug_log .= "Customer Name: '$customer_name'\n";
$debug_log .= "Customer Email: '$customer_email'\n";
$debug_log .= "Title: '$title'\n";
$debug_log .= "Description: '$description'\n";
$debug_log .= "Items: " . json_encode($items) . "\n\n";

// Process misc items
$debug_log .= "=== PROCESSING MISC ITEMS ===\n";
$item_details = [];
$subtotal = 0;

foreach ($items as $index => $item) {
    $name = trim($item['name'] ?? 'Unknown Item');
    $quantity = (int)($item['quantity'] ?? 1);
    $price = (float)($item['price'] ?? 0);
    $line_total = $price * $quantity;
    $subtotal += $line_total;
    
    $debug_log .= "\nMISC ITEM $index:\n";
    $debug_log .= "Name: '$name'\n";
    $debug_log .= "Quantity: $quantity\n";
    $debug_log .= "Price: $price\n";
    $debug_log .= "Line total: $line_total\n";
    
    $item_details[] = [
        'name' => $name,
        'quantity' => $quantity,
        'price' => $price,
        'total' => $line_total
    ];
}

$debug_log .= "\nMISC SUBTOTAL: $subtotal\n";
$debug_log .= "MISC TOTAL: $total\n";

// Create misc order log entry
$misc_entry = [
    'timestamp' => date('Y-m-d H:i:s'),
    'transaction_id' => $transaction_id,
    'customer_name' => $customer_name,
    'customer_email' => $customer_email,
    'title' => $title,
    'description' => $description,
    'items' => $item_details,
    'subtotal' => $subtotal,
    'total' => $total,
    'type' => 'misc_only'
];

$debug_log .= "\n=== WOULD LOG TO misc_orders.log ===\n";
$debug_log .= json_encode($misc_entry, JSON_PRETTY_PRINT) . "\n";

// Write debug log
file_put_contents('misc_order_debug.log', $debug_log);

// Actually write to misc orders log (this is safe since it's just a file)
file_put_contents('misc_orders.log', json_encode($misc_entry) . "\n", FILE_APPEND);

// Return success response
echo json_encode([
    'success' => true,
    'debug_mode' => true,
    'transaction_id' => $transaction_id,
    'total' => $total,
    'items_processed' => count($item_details),
    'message' => 'Misc order logged successfully. Check misc_order_debug.log for details.'
]);
?>