<?php
// box/api/record_order.php - osConcert Database Adapter (WORKING VERSION)
header('Content-Type: application/json');


$config = include __DIR__ . '/../config/config.php';

$conn = new mysqli($config['db_host'], $config['db_user'], $config['db_pass'], $config['db_name']);

if ($conn->connect_error) {
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed']);
    exit;
}

// Initialize debug log
$debug_log = "=== ORDER CREATION DEBUG LOG - " . date('Y-m-d H:i:s') . " ===\n";

// Get JSON input from tablet
$data = json_decode(file_get_contents('php://input'), true);
$debug_log .= "INPUT DATA: " . json_encode($data, JSON_PRETTY_PRINT) . "\n\n";

// Validate required fields
if (!$data || !isset($data['items'], $data['total_amount'], $data['payment_intent_id'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid request data']);
    exit;
}

// Extract order data
$items = $data['items'];
$total = (float)$data['total_amount'];
$payment_intent_id = trim($data['payment_intent_id']);
$customer_name = trim($data['customer_name'] ?? '');
$customer_email = trim($data['customer_email'] ?? '');

$debug_log .= "EXTRACTED DATA:\n";
$debug_log .= "Total: $total\n";
$debug_log .= "Payment Intent: $payment_intent_id\n";
$debug_log .= "Customer Name: '$customer_name'\n";
$debug_log .= "Customer Email: '$customer_email'\n";
$debug_log .= "Items: " . json_encode($items) . "\n\n";

// Handle customer logic
$existing_customer = null;
$customer_id = null;

if ($customer_email) {
    $debug_log .= "LOOKING UP EXISTING CUSTOMER BY EMAIL...\n";
    $stmt = $conn->prepare("
        SELECT c.customers_id, c.customers_firstname, c.customers_lastname, c.customers_email_address, 
               c.customers_telephone, c.customers_default_address_id,
               a.entry_street_address, a.entry_city, a.entry_state, a.entry_postcode, a.entry_country_id
        FROM customers c 
        LEFT JOIN address_book a ON c.customers_id = a.customers_id AND c.customers_default_address_id = a.address_book_id
        WHERE c.customers_email_address = ? 
        LIMIT 1
    ");
    $stmt->bind_param("s", $customer_email);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $existing_customer = $row;
        $customer_id = $row['customers_id'];
        
        // For existing customers: box office sale with real customer billing
        $customer_display_name = 'box office';
        $billing_name = trim($row['customers_firstname'] . ' ' . $row['customers_lastname']);
        $delivery_name = 'Box Office';
        
        // Use existing customer's address data
        $customer_telephone = $row['customers_telephone'] ?? '';
        $customer_street = $row['entry_street_address'] ?? '';
        $customer_city = $row['entry_city'] ?? '';
        $customer_state = $row['entry_state'] ?? '';
        $customer_postcode = $row['entry_postcode'] ?? '';
        $customer_country = 'United States';
        
        $debug_log .= "FOUND EXISTING CUSTOMER: ID=$customer_id\n";
        $debug_log .= "Display Name: '$customer_display_name' (marks as box office sale)\n";
        $debug_log .= "Billing Name: '$billing_name' (real customer)\n";
        $debug_log .= "Delivery Name: '$delivery_name'\n";
        $debug_log .= "Address: $customer_street, $customer_city, $customer_state $customer_postcode\n";
        $debug_log .= "Phone: $customer_telephone\n";
    }
    $stmt->close();
}

if (!$existing_customer) {
    $debug_log .= "NO EXISTING CUSTOMER FOUND - WILL CREATE NEW\n";
    if ($customer_name) {
        $name_parts = explode(' ', $customer_name, 2);
        $firstname = $name_parts[0];
        $lastname = isset($name_parts[1]) ? $name_parts[1] : '';
        $customer_display_name = 'box office';
        $billing_name = $customer_name;
        $delivery_name = 'Box Office';
    } else {
        $firstname = 'Box';
        $lastname = 'Office';
        $customer_display_name = 'box office';
        $billing_name = 'Box Office';
        $delivery_name = 'Box Office';
    }
    
    // Set defaults for new customers
    $customer_telephone = '';
    $customer_street = '';
    $customer_city = '';
    $customer_state = '';
    $customer_postcode = '';
    $customer_country = 'United States';
    
    $debug_log .= "NEW CUSTOMER: firstname='$firstname', lastname='$lastname'\n";
}

// Create new customer if not found
if (!$existing_customer) {
    $customer_sql = "INSERT INTO customers (
        customers_firstname, customers_lastname, customers_email_address,
        customers_telephone, customers_dob, customers_gender, customers_newsletter,
        date_account_created, customers_default_address_id
    ) VALUES (?, ?, ?, ?, '0000-00-00', '', '0', NOW(), 0)";
    
    $stmt = $conn->prepare($customer_sql);
    $stmt->bind_param("ssss", 
        $firstname, $lastname, $customer_email, ''
    );
    $stmt->execute();
    $customer_id = $stmt->insert_id;
    $stmt->close();
    
    $debug_log .= "CREATED NEW CUSTOMER: ID=$customer_id\n";
}

// Create main order record
$order_sql = "INSERT INTO orders (
    customers_id, customers_name, customers_email_address, customers_telephone,
    customers_street_address, customers_city, customers_postcode, customers_state, 
    customers_country, delivery_name, delivery_street_address, delivery_city,
    delivery_postcode, delivery_state, delivery_country, billing_name,
    billing_street_address, billing_city, billing_postcode, billing_state,
    billing_country, payment_method, date_purchased, last_modified,
    orders_status, currency, currency_value, payment_info
) VALUES (
    ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 
    'cash_pos', NOW(), NOW(), 3, 'USD', 1.0, ?
)";

$stmt = $conn->prepare($order_sql);
$stmt->bind_param("isssssssssssssssssssss",
    $customer_id, $customer_display_name, $customer_email, $customer_telephone,
    $customer_street, $customer_city, $customer_postcode, $customer_state, $customer_country,
    $delivery_name, $customer_street, $customer_city, $customer_postcode, $customer_state, $customer_country,
    $billing_name, $customer_street, $customer_city, $customer_postcode, $customer_state, $customer_country,
    $payment_intent_id
);

$stmt->execute();
$order_id = $stmt->insert_id;
$stmt->close();

$debug_log .= "CREATED ORDER: ID=$order_id\n";

// Process each item
$subtotal = 0;
$debug_log .= "\n=== PROCESSING ITEMS ===\n";

foreach ($items as $index => $item) {
    $product_id = (int)$item['ticket_product_id']; 
    $quantity = (int)$item['quantity'];
    
    $debug_log .= "\nITEM $index:\n";
    $debug_log .= "product_id: $product_id\n";
    $debug_log .= "quantity: $quantity\n";
    
    // Get actual product details
    $product_query = "SELECT pd.products_name, p.products_price, p.products_quantity 
                      FROM products p 
                      JOIN products_description pd ON p.products_id = pd.products_id 
                      WHERE p.products_id = ? AND pd.language_id = 1";
    $stmt = $conn->prepare($product_query);
    $stmt->bind_param("i", $product_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $product = $result->fetch_assoc();
    $stmt->close();
    
    if (!$product) {
        $debug_log .= "ERROR: Product ID $product_id not found!\n";
        continue;
    }
    
    $product_name = $product['products_name'];
    $product_price = (float)$product['products_price'];
    $current_quantity = (int)$product['products_quantity'];
    $line_total = $product_price * $quantity;
    $subtotal += $line_total;
    
    $debug_log .= "Product found: '$product_name'\n";
    $debug_log .= "Price: $product_price\n";
    $debug_log .= "Current inventory: $current_quantity\n";
    $debug_log .= "Line total: $line_total\n";
    
    // Insert order_products record
    $order_products_sql = "INSERT INTO orders_products (
        orders_id, products_id, products_name, products_price, 
        final_price, products_quantity, products_tax
    ) VALUES (?, ?, ?, ?, ?, ?, 0)";
    
    $stmt = $conn->prepare($order_products_sql);
    $stmt->bind_param("iissdi", 
        $order_id, $product_id, $product_name, 
        $product_price, $product_price, $quantity
    );
    $stmt->execute();
    $stmt->close();
    
    // Update product inventory
    $update_inventory = "UPDATE products SET products_quantity = products_quantity - ? WHERE products_id = ?";
    $stmt = $conn->prepare($update_inventory);
    $stmt->bind_param("ii", $quantity, $product_id);
    $stmt->execute();
    $stmt->close();
    
    $debug_log .= "INSERTED order_products and updated inventory\n";
}

$debug_log .= "\nSUBTOTAL: $subtotal\n";
$debug_log .= "TOTAL: $total\n";

// Handle mixed orders - log misc items if total > subtotal
if ($total > $subtotal) {
    $misc_amount = $total - $subtotal;
    $debug_log .= "\n=== MIXED ORDER DETECTED ===\n";
    $debug_log .= "Misc amount: $misc_amount\n";
    
    $misc_entry = [
        'order_id' => $order_id,
        'date' => date('Y-m-d H:i:s'),
        'customer' => $billing_name,
        'items' => 'unknown misc items',
        'amount' => $misc_amount,
        'payment_method' => $payment_intent_id,
        'note' => 'Mixed order - misc details lost due to iPad split',
        'type' => 'mixed_order_extras'
    ];
    
    // Actually write to misc_orders.log
    file_put_contents('misc_orders.log', json_encode($misc_entry) . "\n", FILE_APPEND);
    
    $debug_log .= "LOGGED TO misc_orders.log:\n";
    $debug_log .= json_encode($misc_entry, JSON_PRETTY_PRINT) . "\n";
}

// Insert order totals - USE SUBTOTAL (tickets only) for osConcert
$order_totals = [
    ['title' => 'Sub-Total:', 'text' => '$' . number_format($subtotal, 2), 'value' => $subtotal, 'code' => 'ot_subtotal', 'sort_order' => 1],
    ['title' => 'Total:', 'text' => '$' . number_format($subtotal, 2), 'value' => $subtotal, 'code' => 'ot_total', 'sort_order' => 99]
];

foreach ($order_totals as $total_item) {
    $total_sql = "INSERT INTO orders_total (orders_id, title, text, value, class, sort_order) VALUES (?, ?, ?, ?, ?, ?)";
    $stmt = $conn->prepare($total_sql);
    $stmt->bind_param("issdsi", 
        $order_id, $total_item['title'], $total_item['text'], 
        $total_item['value'], $total_item['code'], $total_item['sort_order']
    );
    $stmt->execute();
    $stmt->close();
}

$debug_log .= "INSERTED order totals\n";

// Insert order status history
$status_sql = "INSERT INTO orders_status_history (orders_id, orders_status_id, date_added, comments) VALUES (?, 3, NOW(), 'Order processed via iPad POS')";
$stmt = $conn->prepare($status_sql);
$stmt->bind_param("i", $order_id);
$stmt->execute();
$stmt->close();

$debug_log .= "INSERTED order status history\n";

$debug_log .= "\n=== ORDER CREATION COMPLETE ===\n";

// Write debug log to file
file_put_contents('order_debug.log', $debug_log);

$conn->close();

// Return success response
echo json_encode([
    'success' => true, 
    'order_id' => $order_id,
    'customer_id' => $customer_id,
    'billing_name' => $billing_name,
    'delivery_name' => $delivery_name,
    'subtotal' => $subtotal,
    'total' => $total,
    'message' => 'Order created successfully in osConcert database!'
]);
?>