<?php
// email_helpers.php

function send_order_confirmation_email(int $order_id): bool {
    include_once '../config/getTopDir.php';
    $topDir = getTopDir();
    $config = include $_SERVER['DOCUMENT_ROOT'] . $topDir . 'config/config.php';
    
    // Use the enhanced email library instead of old send_email.php
    require_once $_SERVER['DOCUMENT_ROOT'] . $topDir . 'email/email_lib.php';

    // Connect to DB
    $conn = new mysqli(
        $config['db_host'],
        $config['db_user'],
        $config['db_pass'],
        $config['db_name']
    );
    if ($conn->connect_error) return false;

    // Retrieve organization settings
    $settings = [];
    $settings_result = $conn->query("SELECT `key`, `value` FROM settings");
    if ($settings_result && $settings_result->num_rows > 0) {
        while ($row = $settings_result->fetch_assoc()) {
            $settings[$row['key']] = $row['value'];
        }
    }

    $org_name    = $settings['organization_name']   ?? 'Your Organization';
    $org_email   = $settings['organization_email']  ?? 'no-reply@example.com';
    $org_phone   = $settings['organization_phone']  ?? '';
    $org_address = $settings['organization_address'] ?? '';

    // Get order and user info
    $stmt = $conn->prepare("SELECT o.id, o.total_amount, u.name, u.email FROM orders o JOIN users u ON o.user_id = u.id WHERE o.id = ?");
    $stmt->bind_param("i", $order_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if (!$result || $result->num_rows === 0) {
        $conn->close();
        return false;
    }
    $order_data = $result->fetch_assoc();

    $customer_name = $order_data['name'];
    $customer_email = $order_data['email'];

    // Get ticket details
    $ticket_query = $conn->prepare("SELECT t.ticket_type AS name, e.name AS event_name, tp.price, COUNT(*) as quantity FROM tickets t JOIN events e ON t.event_id = e.id JOIN ticket_products tp ON tp.event_id = e.id AND tp.name = t.ticket_type WHERE t.order_id = ? GROUP BY t.ticket_type, e.name, tp.price");
    $ticket_query->bind_param("i", $order_id);
    $ticket_query->execute();
    $ticket_result = $ticket_query->get_result();

    $tickets = [];
    while ($row = $ticket_result->fetch_assoc()) {
        $tickets[] = $row;
    }

    // Pull donation and discount from orders table
    $stmt = $conn->prepare("SELECT donation, discount_code, discount_amount FROM orders WHERE id = ?");
    $stmt->bind_param("i", $order_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $extras = $result->fetch_assoc();
    
    $conn->close();

    // Compose the email
    $to = $customer_email;
    $subject = "🎟 Your $org_name Tickets Are Ready!";

    $message = "Hello $customer_name,\n\n";
    $message .= "Thank you for your order with $org_name!\n\n";
    $message .= "🧾 Here's a summary of your purchase:\n\n";

    // Group tickets by event
    $event_tickets = [];
    foreach ($tickets as $ticket) {
        $event = $ticket['event_name'] ?? 'Unknown Event';
        if (!isset($event_tickets[$event])) $event_tickets[$event] = [];
        $event_tickets[$event][] = $ticket;
    }

    foreach ($event_tickets as $event => $items) {
        $message .= "🎭 Event: $event\n";
        foreach ($items as $ticket) {
            $line_total = $ticket['price'] * $ticket['quantity'];
            $message .= "- {$ticket['name']} x{$ticket['quantity']} @ $" . number_format($ticket['price'], 2) . " = $" . number_format($line_total, 2) . "\n";
        }
        $message .= "\n";
    }

    if (!empty($extras['discount_code'])) {
        $message .= "💸 Coupon Applied: {$extras['discount_code']}\n";
        $message .= "Discount: -$" . number_format($extras['discount_amount'], 2) . "\n";
    }

    if (!empty($extras['donation'])) {
        $message .= "🙏 Donation: $" . number_format($extras['donation'], 2) . "\n";
    }

    $total_paid = $order_data['total_amount'];
    $message .= "------------------------\n";
    $message .= "Total Paid: $" . number_format($total_paid, 2) . "\n";
    $message .= "------------------------\n\n";

    $message .= "🎟 Download your tickets here:\n";
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $message .= "$protocol://" . $_SERVER['HTTP_HOST'] . $topDir . "tickets/generate_ticket.php?order_id=$order_id\n\n";

    $message .= "We look forward to seeing you at the event!\n\n";
    $message .= "- $org_name Box Office\n";
    if (!empty($org_phone))   $message .= "Phone: $org_phone\n";
    if (!empty($org_address)) $message .= "Address: $org_address\n";

    // Send the email using the enhanced email library
    $result = sendEmail($to, $subject, $message);
    
    return ($result === true);
}

// Additional helper function for sending notification emails to admins
function send_admin_notification_email(int $order_id, string $notification_type = 'new_order'): bool {
    include_once '../config/getTopDir.php';
    $topDir = getTopDir();
    $config = include $_SERVER['DOCUMENT_ROOT'] . $topDir . 'config/config.php';
    require_once $_SERVER['DOCUMENT_ROOT'] . $topDir . 'email/email_lib.php';

    // Connect to DB
    $conn = new mysqli(
        $config['db_host'],
        $config['db_user'],
        $config['db_pass'],
        $config['db_name']
    );
    if ($conn->connect_error) return false;

    // Get organization settings
    $settings = [];
    $settings_result = $conn->query("SELECT `key`, `value` FROM settings");
    if ($settings_result && $settings_result->num_rows > 0) {
        while ($row = $settings_result->fetch_assoc()) {
            $settings[$row['key']] = $row['value'];
        }
    }

    $org_name = $settings['organization_name'] ?? 'Your Organization';
    $admin_email = $settings['organization_email'] ?? 'admin@example.com';

    // Get order details
    $stmt = $conn->prepare("SELECT o.id, o.total_amount, o.created_at, u.name, u.email FROM orders o JOIN users u ON o.user_id = u.id WHERE o.id = ?");
    $stmt->bind_param("i", $order_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if (!$result || $result->num_rows === 0) {
        $conn->close();
        return false;
    }
    $order_data = $result->fetch_assoc();
    $conn->close();

    $to = $admin_email;
    $subject = "🎫 New Order #$order_id - $org_name";
    
    $message = "New order received!\n\n";
    $message .= "Order ID: #$order_id\n";
    $message .= "Customer: {$order_data['name']} ({$order_data['email']})\n";
    $message .= "Amount: $" . number_format($order_data['total_amount'], 2) . "\n";
    $message .= "Date: {$order_data['created_at']}\n\n";
    $message .= "View order details in the admin panel.\n";

    $result = sendEmail($to, $subject, $message);
    return ($result === true);
}

// Helper function for sending custom emails (useful for other parts of the system)
function send_custom_email(string $to, string $subject, string $message, bool $isHTML = false): bool {
    include_once '../config/getTopDir.php';
    $topDir = getTopDir();
    require_once $_SERVER['DOCUMENT_ROOT'] . $topDir . 'email/email_lib.php';
    
    $result = sendEmail($to, $subject, $message, $isHTML);
    return ($result === true);
}
?>