<?php
// email_lib_config.php - Config-based version for osConcert integration
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

function sendEmail($to, $subject, $body, $isHTML = false, $config_file = null) {
    ini_set('display_errors', 1);
    error_reporting(E_ALL);
    
    $autoload = __DIR__ . '/vendor/autoload.php'; 
    if (!file_exists($autoload)) {
        return "❌ Could not find autoload.php at $autoload";
    }
    require_once $autoload;
    
    // Load email configuration from config file
    if ($config_file && file_exists($config_file)) {
        $email_config = include $config_file;
    } else {
        // Default config file location
        $default_config = __DIR__ . '/../config/email_config.php';
        if (!file_exists($default_config)) {
            return "❌ Email configuration file not found";
        }
        $email_config = include $default_config;
    }
    
    // Validate email
    if (!$to || !filter_var($to, FILTER_VALIDATE_EMAIL)) {
        return "❌ Invalid or missing 'to' address.";
    }
    
    // Check if SMTP is enabled
    if (empty($email_config['smtp_enabled']) || $email_config['smtp_enabled'] === false) {
        // Use PHP mail() as fallback
        $from_email = $email_config['smtp_from_email'] ?? $email_config['organization_email'] ?? 'no-reply@localhost';
        $from_name = $email_config['smtp_from_name'] ?? $email_config['organization_name'] ?? 'System';
        
        $headers = "From: " . $from_name . " <" . $from_email . ">\r\n";
        $headers .= "Reply-To: " . $from_email . "\r\n";
        $headers .= "X-Mailer: PHP/" . phpversion() . "\r\n";
        
        if ($isHTML) {
            $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        } else {
            $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
        }
        
        if (mail($to, $subject, $body, $headers)) {
            return true;
        } else {
            return "❌ PHP mail() failed to send email.";
        }
    }
    
    // Use SMTP with PHPMailer
    $mail = new PHPMailer(true);
    
    try {
        // Server settings
        $mail->isSMTP();
        $mail->Host = $email_config['smtp_host'] ?? '';
        $mail->Port = intval($email_config['smtp_port'] ?? 587);
        
        // Security settings
        $smtp_secure = $email_config['smtp_secure'] ?? 'tls';
        if ($smtp_secure === 'ssl') {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
        } elseif ($smtp_secure === 'tls') {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
        }
        // If 'none' or empty, don't set SMTPSecure
        
        // Authentication
        $smtp_auth = $email_config['smtp_auth'] ?? true;
        if ($smtp_auth === true || $smtp_auth === '1') {
            $mail->SMTPAuth = true;
            $mail->Username = $email_config['smtp_user'] ?? '';
            $mail->Password = $email_config['smtp_pass'] ?? '';
        } else {
            $mail->SMTPAuth = false;
        }
        
        // Timeout setting
        $mail->Timeout = intval($email_config['smtp_timeout'] ?? 30);
        
        // Debug level (only if explicitly set > 0)
        $debug_level = intval($email_config['smtp_debug'] ?? 0);
        if ($debug_level > 0) {
            $mail->SMTPDebug = $debug_level;
            $mail->Debugoutput = 'error_log'; // Send debug to error log instead of output
        }
        
        // Email format
        $mail->isHTML($isHTML);
        $mail->CharSet = 'UTF-8';
        
        // Headers
        $mail->addCustomHeader('X-Mailer', 'PHPMailer ' . PHPMailer::VERSION);
        $mail->addCustomHeader('X-Priority', '3'); // Normal priority
        $mail->addCustomHeader('X-MSMail-Priority', 'Normal');
        
        // From settings
        $from_email = $email_config['smtp_from_email'] ?? $email_config['organization_email'] ?? 'no-reply@localhost';
        $from_name = $email_config['smtp_from_name'] ?? $email_config['organization_name'] ?? 'System';
        
        $mail->setFrom($from_email, $from_name);
        $mail->addAddress($to);
        $mail->Subject = $subject;
        $mail->Body = $body;
        
        $mail->send();
        return true;
        
    } catch (Exception $e) {
        return "❌ SMTP Error: " . $mail->ErrorInfo;
    }
}

// Enhanced sendEmail function with more options for config-based system
function sendEmailAdvanced($to, $subject, $body, $options = [], $config_file = null) {
    // Extract options with defaults
    $isHTML = $options['isHTML'] ?? false;
    $cc = $options['cc'] ?? [];
    $bcc = $options['bcc'] ?? [];
    $attachments = $options['attachments'] ?? [];
    $replyTo = $options['replyTo'] ?? null;
    $priority = $options['priority'] ?? 3; // 1=High, 3=Normal, 5=Low
    
    ini_set('display_errors', 1);
    error_reporting(E_ALL);
    
    $autoload = __DIR__ . '/vendor/autoload.php'; 
    if (!file_exists($autoload)) {
        return "❌ Could not find autoload.php at $autoload";
    }
    require_once $autoload;
    
    // Load email configuration from config file
    if ($config_file && file_exists($config_file)) {
        $email_config = include $config_file;
    } else {
        // Default config file location
        $default_config = __DIR__ . '/email_config.php';
        if (!file_exists($default_config)) {
            return "❌ Email configuration file not found";
        }
        $email_config = include $default_config;
    }
    
    // Validate primary email
    if (!$to || !filter_var($to, FILTER_VALIDATE_EMAIL)) {
        return "❌ Invalid or missing 'to' address.";
    }
    
    // For advanced features, we need SMTP
    if (empty($email_config['smtp_enabled']) || $email_config['smtp_enabled'] === false) {
        // Fall back to simple sendEmail for basic functionality
        return sendEmail($to, $subject, $body, $isHTML, $config_file);
    }
    
    $mail = new PHPMailer(true);
    
    try {
        // Server settings
        $mail->isSMTP();
        $mail->Host = $email_config['smtp_host'] ?? '';
        $mail->Port = intval($email_config['smtp_port'] ?? 587);
        
        // Security settings
        $smtp_secure = $email_config['smtp_secure'] ?? 'tls';
        if ($smtp_secure === 'ssl') {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
        } elseif ($smtp_secure === 'tls') {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
        }
        
        // Authentication
        $smtp_auth = $email_config['smtp_auth'] ?? true;
        if ($smtp_auth === true || $smtp_auth === '1') {
            $mail->SMTPAuth = true;
            $mail->Username = $email_config['smtp_user'] ?? '';
            $mail->Password = $email_config['smtp_pass'] ?? '';
        } else {
            $mail->SMTPAuth = false;
        }
        
        // Timeout setting
        $mail->Timeout = intval($email_config['smtp_timeout'] ?? 30);
        
        // Debug level (only if explicitly set > 0)
        $debug_level = intval($email_config['smtp_debug'] ?? 0);
        if ($debug_level > 0) {
            $mail->SMTPDebug = $debug_level;
            $mail->Debugoutput = 'error_log';
        }
        
        // Email format
        $mail->isHTML($isHTML);
        $mail->CharSet = 'UTF-8';
        
        // Priority headers
        $mail->addCustomHeader('X-Mailer', 'PHPMailer ' . PHPMailer::VERSION);
        $mail->addCustomHeader('X-Priority', (string)$priority);
        if ($priority == 1) {
            $mail->addCustomHeader('X-MSMail-Priority', 'High');
        } elseif ($priority == 5) {
            $mail->addCustomHeader('X-MSMail-Priority', 'Low');
        } else {
            $mail->addCustomHeader('X-MSMail-Priority', 'Normal');
        }
        
        // From settings
        $from_email = $email_config['smtp_from_email'] ?? $email_config['organization_email'] ?? 'no-reply@localhost';
        $from_name = $email_config['smtp_from_name'] ?? $email_config['organization_name'] ?? 'System';
        
        $mail->setFrom($from_email, $from_name);
        
        // Reply-To
        if ($replyTo && filter_var($replyTo, FILTER_VALIDATE_EMAIL)) {
            $mail->addReplyTo($replyTo);
        }
        
        // Recipients
        $mail->addAddress($to);
        
        // CC recipients
        if (is_array($cc)) {
            foreach ($cc as $email) {
                if (filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    $mail->addCC($email);
                }
            }
        }
        
        // BCC recipients
        if (is_array($bcc)) {
            foreach ($bcc as $email) {
                if (filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    $mail->addBCC($email);
                }
            }
        }
        
        // Attachments
        if (is_array($attachments)) {
            foreach ($attachments as $attachment) {
                if (is_string($attachment) && file_exists($attachment)) {
                    $mail->addAttachment($attachment);
                } elseif (is_array($attachment) && isset($attachment['path']) && file_exists($attachment['path'])) {
                    $name = $attachment['name'] ?? basename($attachment['path']);
                    $mail->addAttachment($attachment['path'], $name);
                }
            }
        }
        
        $mail->Subject = $subject;
        $mail->Body = $body;
        
        $mail->send();
        return true;
        
    } catch (Exception $e) {
        return "❌ SMTP Error: " . $mail->ErrorInfo;
    }
}

// HTTP endpoint for direct email sending (config-based)
if (basename($_SERVER['PHP_SELF']) == 'email_lib_config.php') {
    $to = $_REQUEST['to'] ?? null;
    $subject = $_REQUEST['subject'] ?? 'No Subject';
    $body = $_REQUEST['body'] ?? 'No message body.';
    $isHTML = isset($_REQUEST['html']) && $_REQUEST['html'] == '1';
    $config_file = $_REQUEST['config'] ?? null;
    
    if (!$to || !filter_var($to, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        echo "❌ Invalid or missing 'to' address.";
        exit;
    }
    
    $result = sendEmail($to, $subject, $body, $isHTML, $config_file);
    
    if ($result === true) {
        echo "✅ Email sent successfully!";
    } else {
        http_response_code(500);
        echo $result;
    }
}
?>